using System;
using System.Collections;
using System.Data;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/17/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Base Business Object most business object layer classes that will persist will inherit from this class
	///</summary>

	#endregion

	[Serializable]  
	public abstract class BaseBusinessObject : 
		ICloneable, 
		IBaseBusinessObject, 
		IDisposable
	{		
		#region variables
		/// <summary>
		/// Indicates that object was disposed.
		/// </summary>
		private bool _isDisposed = false;
		///     <summary>
		///     Is the record read only
		///     </summary>
		private bool _isReadOnly;
		///     <summary>              
		///     Comments               
		///     </summary>             
		private string _comments = String.Empty;
		/// <summary>
		/// Is the object new
		/// </summary>
		private bool _isNew = false;
		/// <summary>
		/// Has the object been changed
		/// </summary>
		private bool _isDirty = false;	
		/// <summary>
		/// Has the textual representation of the record status code
		/// </summary>
		private string _recordStatusText;
		///	<summary>
		///	The logon id of the user who performed the last update.
		///	</summary>
		private string	_lastUpdateUser = String.Empty;
		///	<summary>
		///	The date and time the record was last updated.
		///	</summary>
		private System.DateTime _lastUpdateDate = DateTime.MinValue;
		/// <summary>
		/// Used to check if someone else has modified your retrieved data
		/// </summary>
		private Byte[] _rowVersion;

		private static Hashtable _brokenRulesValidationMessages;
		private ArrayList _instanceBrokenRules;
		private Hashtable _initiallySetBrokenRules;
		private string _workingFormName, _workingRuleSet;

		#endregion		

		static BaseBusinessObject()
		{
			LoadBrokenRulesValidationMessages();
		}
		
		/// <summary>
		/// Disposed
		/// </summary>
		protected bool Disposed
		{
			get
			{
				lock(this)
				{
					return _isDisposed;
				}
			}
		}
		

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2319"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2320"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Do not make Dispose() virtual - prevent subclass from overriding
		/// </summary>
		public void Dispose()
		{
			lock(this)
			{
				if(_isDisposed == false)
				{
					_isDisposed = true; 
					GC.SuppressFinalize(this);
				}
			}
		}

		///<Developers>
		///	<Developer>Gregory Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2323"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2332"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ObjectDisposedException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns an error if the object has been disposed
		/// </summary>
		public void CheckIfDisposed()
		{
			if (Disposed) //verify in every method
			{ 
				throw new ObjectDisposedException( "Object is already disposed" );
			}
		}

		#region events

		/// <summary>
		/// IsDirtyChanged Event Handler
		/// </summary>
		public event EventHandler IsDirtyChanged;

		/// <summary>
		/// onValid Event Handler
		/// </summary>
		public System.EventHandler BaseOnValid;

		/// <summary>
		/// onInvalid Event Handler
		/// </summary>
		public System.EventHandler BaseOnInvalid;
		
		/// <summary>
		/// IsDirtyChanged
		/// </summary>
		protected virtual void OnIsDirtyChanged()
		{
			if(IsDirtyChanged != null)
				IsDirtyChanged(this, EventArgs.Empty);
		}
        
		/// <summary>
		/// onValid Event
		/// </summary>
		public event EventHandler Valid
		{
			add 
			{
				BaseOnValid += value;
			}
			remove 
			{ 
				BaseOnValid -= value;
			}
		}

		/// <summary>
		/// onInvalid Event
		/// </summary>
		public event EventHandler Invalid
		{
			add 
			{
				BaseOnInvalid += value;
			}
			remove 
			{ 
				BaseOnInvalid -= value;
			}
		}

		/// <newpara> Function Name: OnValidWorklist</newpara>
		/// <newpara> Developers: Cameron Taylor</newpara>
		/// <newpara> Site Name: Hines OIFO </newpara>
		/// <newpara> Date Created: 7/5/2002</newpara>
		/// <summary> Description: Virtual component ValidWorklist event</summary>
		/// <param name="sender">Object</param>
		/// <param name="e">EventArgs</param>
		/// <returns>N/A</returns>
		protected virtual void OnValid(object sender,System.EventArgs e)
		{
		}

		/// <newpara> Function Name: OnInvalidWorklist</newpara>
		/// <newpara> Developers: Cameron Taylor</newpara>
		/// <newpara> Site Name: Hines OIFO </newpara>
		/// <newpara> Date Created: 7/5/2002</newpara>
		/// <summary> Description: Virtual component InvalidWorklist event</summary>
		/// <param name="sender">Object</param>
		/// <param name="e">EventArgs</param>
		/// <returns>N/A</returns>
		protected virtual void OnInvalid(object sender,System.EventArgs e)
		{
		}

		#endregion	
       		
		/// <summary>
		/// If rule is broken, adds it to collection of broken rules,
		/// otherwise removes it.
		/// </summary>
		/// <param name="rule">
		///		Broken rule ID. May contain underscore character - then only part 
		///		after underscore will be used
		///	</param>
		/// <param name="isBroken">Specifies if rule is broken</param>
		protected bool RuleBroken( string rule, bool isBroken )
		{
			EnsureBrokenRulesCollectionsExist();

			string ruleName = GetRuleNameFromAccessorName( rule );
			
			if( isBroken )
				AddUpdateBrokenRule( ruleName );								
			else
				RemoveBrokenRule( ruleName );

			return isBroken;
		}

		/// <summary>
		/// If rule is broken, adds it to collection of broken rules,
		/// otherwise removes it.
		/// 
		/// WARNING: if error message is not empty the method will 
		/// throw <see cref="BrokenRuleException"/>.
		/// </summary>
		/// <param name="rule">
		///		Broken rule ID. May contain underscore character - then only part 
		///		after underscore will be used
		///	</param>
		/// <param name="isBroken">Specifies if rule is broken</param>
		/// <param name="errorMessage">Error Provider Message</param>
		protected bool RuleBroken(string rule, bool isBroken, string errorMessage)
		{
			if( !RuleBroken( rule, isBroken ) )
				return false;

			if( errorMessage != String.Empty )
				throw new BrokenRuleException( errorMessage );

			return true;
		}

		/// <summary>
		/// Extracts rule name from accessor property name returned by reflection methods.
		/// I.e. for "set_UnitMedia" - it will return "UnitMedia". 
		/// Basically it will return all characters following the first underscore 
		/// character or the whole string if no underscore was found.
		/// </summary>
		/// <param name="accessorName">Rule name with get_/set_ prefix</param>
		/// <returns>Rule name</returns>
		private string GetRuleNameFromAccessorName( string accessorName )
		{
			return accessorName.Substring(accessorName.IndexOf("_") + 1);
		}

		/// <summary>
		/// Checks if object instance collections of broken rules are
		/// initialized. If not - creates collections.
		/// </summary>
		private void EnsureBrokenRulesCollectionsExist()
		{
			if( _instanceBrokenRules == null ) 
				_instanceBrokenRules = new ArrayList();

			if( _initiallySetBrokenRules == null ) 
				_initiallySetBrokenRules = new Hashtable();
		}

		/// <summary>
		/// Adds broken rule to collection and notifies clients.
		/// </summary>
		/// <param name="rule">Broken rule ID</param>
		private void AddUpdateBrokenRule( string rule )
		{
			EnsureBrokenRulesCollectionsExist();

			if( !_instanceBrokenRules.Contains( rule ) && _initiallySetBrokenRules.ContainsKey( rule ) )
				_instanceBrokenRules.Add( rule );

			NotifyClientsAboutStateChange();
		}

		/// <summary>
		/// Removes broken rule from collection.
		/// Notifies clients when there are
		/// no more broken rules.
		/// </summary>
		/// <param name="rule">Broken rule ID</param>
		private void RemoveBrokenRule(string rule)
		{
			_instanceBrokenRules.Remove( rule );
			
			NotifyClientsAboutStateChange();
		}

		///<Developers>
		///	<Developer>Gregory Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2348"> 
		///		<ExpectedInput>Form, rule name</ExpectedInput>
		///		<ExpectedOutput>Validation Text</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2355"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets validation error message for a single broken rule.
		/// WARNING: rule does not have to be broken! The method will return a message anyway!
		/// </summary>
		/// <param name="formClassName">Form or class name.</param>
		/// <param name="ruleName">Broken rule name.</param>
		/// <returns>Validation error message if the rule is found. Otherwise - null.</returns>
		public string GetSingleBrokenRuleMessage( string formClassName, string ruleName )
		{
			if( formClassName == null )
				throw( new ArgumentNullException( "formName" ) );

			if( ruleName == null )
				throw( new ArgumentNullException( "ruleName" ) );
			
			return (string)_brokenRulesValidationMessages[ formClassName + GetRuleNameFromAccessorName( ruleName ) ];
		}

		///<Developers>
		///	<Developer>Gregory Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2356"> 
		///		<ExpectedInput>rule name</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2359"> 
		///		<ExpectedInput>invalid rulename</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks whether specific business rule is broken.
		/// </summary>
		/// <param name="rule">Broken rule name.</param>
		/// <returns>True if the specified rule is broken. Otherwise - false.</returns>
		public bool IsRuleBroken( string rule )
		{		
			if( _instanceBrokenRules == null ) 
				return( false );
			
			return _instanceBrokenRules.Contains( GetRuleNameFromAccessorName( rule ) );
		}
		
		/// <summary>
		/// Verifies that internal static Hashtable containing all validation 
		/// messages is created and filled with data. If not - creates Hashtable
		/// and populates it with data from database.
		/// </summary>
		private static void LoadBrokenRulesValidationMessages()
		{
			lock( typeof( BaseBusinessObject ) )
			{
				if( _brokenRulesValidationMessages != null ) 
					return;
				else
					_brokenRulesValidationMessages = new Hashtable(200);
			
				DataTable dt = DAL.BrokenRules.GetBrokenRulesValidationMessages();

				foreach( DataRow dr in dt.Rows )
					_brokenRulesValidationMessages.Add( dr[Common.VbecsTables.BrokenRule.FormName].ToString() + 
						dr[Common.VbecsTables.BrokenRule.FieldName].ToString(), 
						dr[Common.VbecsTables.BrokenRule.ValidationMessage].ToString() );
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1582"> 
		///		<ExpectedInput>A string representation of the form name</ExpectedInput>
		///		<ExpectedOutput>A string representation of all broken rules validations</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2362"> 
		///		<ExpectedInput>invalid formname</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// A single string representation of broken rules validation messages
		/// </summary>
		/// <param name="formName">Form name</param>
		/// <returns></returns>
		public virtual string GetBrokenRulesMessage( string formName )
		{
			EnsureBrokenRulesCollectionsExist();			

			System.Text.StringBuilder sb = new System.Text.StringBuilder();			
			bool isFirstItemSet = false;

			foreach( string rule in _instanceBrokenRules )
			{
				if(isFirstItemSet )	
					sb.Append( Environment.NewLine );

				string msg = (string)_brokenRulesValidationMessages[formName + rule];

				if (msg != null)
				{
					isFirstItemSet = true;
					sb.Append(msg);
				}
			}

			return isFirstItemSet ? sb.ToString() : String.Empty;
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9188"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9189"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985
		/// A single string representation of broken rules validation messages
		/// with no new lines inserted between rules
		/// </summary>
		/// <param name="formName">Form name</param>
		/// <returns></returns>
		public virtual string GetBrokenRulesMessageSingleLine( string formName )
		{
			EnsureBrokenRulesCollectionsExist();			

			System.Text.StringBuilder sb = new System.Text.StringBuilder( string.Empty );			

			foreach( string rule in _instanceBrokenRules )
			{
				string msg = (string)_brokenRulesValidationMessages[formName + rule];

				if ( msg != null )
				{
					if( sb.Length > 0 )	
						sb.Append( " " );

					sb.Append( msg );
				}
			}

			return sb.ToString();
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="709"> 
		///		<ExpectedInput>Form name that the broken rules apply to</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount > 0</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2865"> 
		///		<ExpectedInput>Invalid FormName</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount = 0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets initial broken rules. Conventional broken rules 
		/// collection is reset as well. 
		/// </summary>
		/// <param name="formName">Form name</param>
		public void SetInitialBrokenRules( string formName)
		{
			SetInitialBrokenRules( formName, "default" );
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2363"> 
		///		<ExpectedInput>Form name, Ruleset</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount > 0</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2364"> 
		///		<ExpectedInput>Invalid FormName, RuleSet</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount = 0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the initial broken rules for an object given a form name and rule set
		/// </summary>
		/// <param name="formName"></param>
		/// <param name="ruleSet"></param>
		public void SetInitialBrokenRules( string formName, string ruleSet )
		{
			SetInitialBrokenRules( formName, ruleSet, false );
		}


		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2365"> 
		///		<ExpectedInput>Form name, Ruleset, bool</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount >= 0</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2366"> 
		///		<ExpectedInput>Invalid FormName, RuleSet</ExpectedInput>
		///		<ExpectedOutput>BrokenRuleCount = 0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Sets the initial broken rules for an object given a form name and rule set
		/// </summary>
		/// <param name="formName"></param>
		/// <param name="ruleSet"></param>
		/// <param name="unbreakRules"></param>
		public void SetInitialBrokenRules( string formName, string ruleSet, bool unbreakRules )
		{
			EnsureBrokenRulesCollectionsExist();

			_instanceBrokenRules.Clear();
			
			if( _workingFormName == formName && _workingRuleSet == ruleSet && _initiallySetBrokenRules.Count > 0 && !unbreakRules )
			{
				// already available in mem
				foreach( string rule in _initiallySetBrokenRules.Keys )
					if( (bool)_initiallySetBrokenRules[rule] )
						_instanceBrokenRules.Add( rule );
			}
			else
			{					
				// fetch from DB
				_initiallySetBrokenRules.Clear();

				DataRow[] _rulesAsRows = DAL.BrokenRules.GetBrokenRules( formName, ruleSet );

				foreach( DataRow dr in _rulesAsRows )
				{
					_initiallySetBrokenRules.Add( dr[Common.VbecsTables.BrokenRule.FieldName].ToString(), 
						Convert.ToBoolean( dr[Common.VbecsTables.BrokenRuleSet.IsInitiallyBroken] ) );

					if( Convert.ToBoolean( dr[Common.VbecsTables.BrokenRuleSet.IsInitiallyBroken] ) && !unbreakRules )
						_instanceBrokenRules.Add( dr[Common.VbecsTables.BrokenRule.FieldName].ToString() );
				}

				_workingFormName = formName;
				_workingRuleSet = ruleSet;
			}

			NotifyClientsAboutStateChange();
		}

		/// <summary>
		///	Triggers the appropriate chain
		///	of event handlers (fires up an event) 
		///	depending on valid/invalid object state.
		/// </summary>
		private void NotifyClientsAboutStateChange()
		{
			if (this.IsValid)
			{
				if (BaseOnValid != null) BaseOnValid(this, EventArgs.Empty);		
			}
			else
			{
				if (BaseOnInvalid != null) BaseOnInvalid(this, EventArgs.Empty);
			}
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/25/2003</CreationDate>
		///
		///<TestCases>
		///	
		///<Case type="0" testid ="1578"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2367"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns if the object is locked or not
		/// </summary>
		public bool IsReadOnly
		{
			get 
			{
				return _isReadOnly;
			}
			set 
			{
				_isReadOnly = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2381"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2377"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///			
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Has the object been changed
		/// </summary>
		public virtual bool IsDirty
		{
			get
			{ 
				return _isDirty; 
			}
			set 
			{ 
				_isDirty = value;
				this.OnIsDirtyChanged();
			}
		}


		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2378"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2387"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///			
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///	Returns a count of the rules that are broken
		/// </summary>
		public virtual int BrokenRulesCount
		{
			get
			{
				return _instanceBrokenRules == null ? 0 : _instanceBrokenRules.Count;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="288"> 
		///		<ExpectedInput>A boolean indicating validity</ExpectedInput>
		///		<ExpectedOutput>A boolean indicating validity</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2388"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Boolean that determines if division object loaded user is valid
		/// </summary>
		public virtual bool IsValid
		{
			get
			{
				return _instanceBrokenRules == null ? true : (_instanceBrokenRules.Count == 0 );
			}
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2382"> 
		///		<ExpectedInput>A boolean</ExpectedInput>
		///		<ExpectedOutput>A boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2390"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Is the object new
		/// </summary>
		public virtual bool IsNew
		{
			get 
			{ 
				return _isNew;
			}
			set 
			{ 
				_isNew = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1580"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2391"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// A generic non required comment
		/// </summary>
		public virtual string Comments
		{
			get
			{
				return _comments;
			}
			set
			{
				_comments = value;
			}
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2383"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2395"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///	<summary>
		///	The logon id of the user who performed the last update., BR_2.13
		///	</summary>
		public string LastUpdateUser
		{
			get
			{
				return _lastUpdateUser;
			}
			set
			{
				_lastUpdateUser = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///
		///<Case type="0" testid ="2384"> 
		///		<ExpectedInput>date</ExpectedInput>
		///		<ExpectedOutput>date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2397"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///	<summary>
		///	The date and time the record was last updated, BR_2.13
		///	</summary>
		public DateTime LastUpdateDate
		{
			get
			{
				return _lastUpdateDate;
			}
			set
			{
				_lastUpdateDate = value;
			}
		}
			
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="400"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2398"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RecordStatustext gets or set the record status text
		/// and raises an event that it has changed
		/// </summary>
		public string RecordStatusText
		{
			get
			{
				return _recordStatusText;
			}
			set
			{
				_recordStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2385"> 
		///		<ExpectedInput>byte array</ExpectedInput>
		///		<ExpectedOutput>byte array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2864"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the rowversion as an array of bytes
		/// </summary>
		public byte[] RowVersion
		{
			get
			{
				return _rowVersion;
			}
			set
			{
				_rowVersion = value;
			}
		}
		//
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1579"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Cloned object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2399"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a memberwise clone
		/// </summary>
		public object Clone()
		{
			return this.MemberwiseClone();
		}

		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		/// <param name="dtRow">data row</param>
		/// <returns></returns>
		public abstract DataRow LoadDataRowFromThis( DataRow dtRow );
		
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		/// <param name="dtRow">data row</param>
		protected abstract void LoadFromDataRow( DataRow dtRow );		


	}
}